# Atelier R
# P.Casadebaig 4 octobre 2012
# Bibliothèques utilisées
library(plyr)
library(reshape)
library(ggplot2)

########################################################################
### Plan
# 1. Importer
# 2. Analyser
# 	2.1 Sélectionner
# 	2.2 Trier
# 	2.3 Traiter

########################################################################

rm(list = ls()) # supprimer tout les objets en memoire
### STEP : 1. Importer

## Depuis des fichiers

# au format texte
wd <- "C:/R_ITA/"
d <- read.table(file=paste(wd, "/data/meteoPC.csv", sep=""), header=T, sep=";", dec=".") # lecture
write.table(d, file=paste(wd, "/data/export.csv", sep=""), sep=";", dec=".", row.names=FALSE) # ecriture

# OPTION import / export de .RData, fichiers au format R

save(d, file=paste(wd, "/data/climate.RData", sep=""), compress="bzip2") # ecriture
load(file=paste(wd, "/data/climate.RData", sep="")) # lecture


## Automatisation de la lecture pour importer des séries de fichiers


## Depuis une base de donnée (?)


########################################################################
### STEP : 2. Analyser

## Sélectionner
# sélectionner toutes les colonnes d'un data.frame sauf celle qui ont le noms A, B ou C
# OPTION : même méthodologie pour les lignes

# basé sur les noms
str(d)
s <- colnames(d) %in% c("TX","TN","TM") != TRUE # vecteur pour la sélection
str(d[,s])

# basé sur les indices
str(d[,-(5:7)])
str(d[,- match(c("TN","TX","TM"), colnames(d))])

# filtrer
ggplot(data=d[(d$TM >= 0 & d$TM <= 10),], aes(x=TM)) + geom_histogram(binwidth=0.1) # et
ggplot(data=d[(d$TX >= 30 | d$TN <= -5),], aes(x=TM)) + geom_histogram(binwidth=0.5) # ou


## Trier

# Trier un data frame selon 1, ou plusieurs colonnes
# OPTIONS : plyr::arrange, structure de donnée data.table
head(d[order(-d$TX),])
head(d[order(-d$TX, -d$TM),]) 
head(with(d, d[order(-TX, -TM),])) # utilisation de with() pour éviter de répéter des noms

# Trier les niveaux d'un facteur
f <- factor(letters[1:4])
f <- factor(f, levels=c("b","a","c","d"))

# Joindre deux tableaux
postes <- data.frame(Code.MF=31069001, site="Blagnac", region="MP", departement=31)
head(merge(postes, d))


## Traiter

# changer le type de données d'une colonne
d$Date <- strptime(d$Date, "%d/%m/%Y")

# ajouter une colonne (informations, variables élaborées...)
d <- cbind(d, PAR = d$GLOT/100 * 0.48) # ajout
d$departement <- NULL # suppression

# ajouter 
# OPTION : utilisation de plyr::mutate (utilisation de colonne itérativement) 
d <- transform(d,
	JD = as.numeric(format(Date, format = "%j")),
	Y = as.numeric(format(Date, format = "%Y")),
	M = as.numeric(format(Date, format = "%m")),
	D = as.numeric(format(Date, format = "%d"))
	)
	
ggplot(aes(x=GLOT), data=d) + geom_histogram() + facet_wrap(~ Y)

# changer de disposition de tableau (sens et ordre des colonnes) : format DB vers analyse
head(xtabs(RR ~ JD + Y, data=d[,c("JD","Y","RR")])) # base::xtabs (simple)
head(cast(d[,c("JD","Y","RR")], JD ~ Y)) # reshape::cast (possible de coupler des opération sur les groupes)
head(reshape(d[,c("JD","Y","RR")], idvar="JD", timevar="Y", direction="wide")) # base::reshape (arguments difficiles à retenir)

# l'inverse, format analyse vers DB (très utile pour les graphiques)
# OPTION stack
dw <- xtabs(RR ~ JD + Y, data=d[,c("JD","Y","RR")]) 
head(melt(dw)) # reshape::melt
ggplot(data=melt(dw), aes(x=JD, y=value)) + geom_line() + facet_wrap(~ Y)

# appliquer une fonction par modalité d'un facteur : {split - apply - combine}

# tableau croisé dynamique, pivoting (plyr)
em <- ddply(d, "Y", summarise, s=sum(GLOT)) # somme de rayonnement par année
ggplot(aes(x=Y, y=s), data=em) + geom_point()
d[d$Y %in% 1975:1991 ==TRUE ,]$GLOT <- d[d$Y %in% 1975:1991 ==TRUE ,]$GLOT * 4.1868

# OPTION : fonction aggregate (utilisation de l'interface "formule")
head(aggregate(GLOT ~ Y, data=d, sum))
head(aggregate(GLOT ~ Y + M, data=d, sum))
head(aggregate(cbind(GLOT, RR) ~ Y + M, data=d, sum))

# plusieurs traitements de sortie
dm <- ddply(d, "Y", summarise, s=sum(GLOT), p=sum(RR), t=mean((TN+TX)/2))
ggplot(dm, aes(x=t, y=p)) + geom_text(aes(label=Y), size=3) 

# cadre plus général
# OPTION apply, lapply
ecriture <- function(x) {write.table(x, file=paste("../data/serie/",unique(x$Y),".csv", sep=""), sep=";", dec=".", row.names=FALSE)} 
ddply(d, "Y", ecriture)

# itération pour produire une liste de différents modèles statistique
model <- function(x) {lm(ETP ~ TX + RR + GLOT, data=x)}
models <- dlply(d, "Y", model) # liste de modèles ajustés
models.coefs <- ldply(models, coef)
ggplot(data=melt(models.coefs[,-2], id.vars="Y"), aes(x=variable, y=value, group=Y)) + geom_line()




