# implement an SEIR model with delays
# use dede function in deSolve for integration and delays
# version with sites instead of proportion

library(deSolve)

#-----------------------------
# the derivative function
# definition of the rates
#-----------------------------
seir <- function(t,y,parms) {

# calculate the infectious number
# here we have start and it stays infectious the 
# entire IP
  if (t < IP)
    inf2 <- start
  else
    inf2 <-0

# these are new infections generated in the model 
# nothing until after one LP
if (t < LP)
    inf1  <- 0 
  else
    inf1  <- lagvalue(t - LP)

# total number infectious is the sum of these two minus removals

# calculate removals
# if we are past the IP the initial infection is removed
  if (t < (IP))
    remove2 <- 0
  else 
    remove2 <- start

# if we are beyond the sum of LP and IP then the new lesions are also removed
  if (t > (LP+IP))
    remove1 <- lagvalue(t-(LP+IP))
  else 
    remove1 <- 0

# the removed area is the sum of these two  
remove <- remove1 + remove2

# infections is the sum of the two components minus removals
infectious <- inf1 + inf2 - remove

# minor problems with roundoff can create negative values (==BAD)
if (infectious <0 )
  infectious <- 0

# y is the total number of NEW infections
# but the infectious and remove have to be subtracted to give the latent ones
latent <- y -  infectious - remove
if (latent < 0) 
  latent <- 0

# y is what we integrate (new infections)
dy <-  Rc * infectious * (1-((y+start)/k)) # like Zadoks 1971 with sites

# k is the total number of sites
# y plus start can never be greate than k

# return the differential, plus the actual change and the state of
# latent, infectious and removed
list(dy, dy=dy, latent=latent,infectious=infectious, removed=remove)

} # end of the definition of what to integrate

##-----------------------------
## parameter values
k <- 10000   # max lesions
LP = 8   # latency period
IP = 1   # infectious period
Rc <- 3  # corrected basic infection rate
# remember that IP*Rc has to be greater than 1
##-----------------------------

##-----------------------------
## initial values and times
start <- 0          # number of initial infectious lesions
yinit <-  c(y = 1)  # initial number of new latent infections
# can have values for either or even both
# for the simulation
step <- 1         # time step
maxtime <- 150      # end of simulation
times <- seq(0, maxtime, by=step)  # we start at zero
##-----------------------------

##-----------------------------
## solve the model 
## dede returns a matrix but it goes directly into a data frame  
## we are integrating new infections, but they become
## infectious or removed via the delay functions
## thus we can start with zero infections if we have some infectious material
## at the beginning (variable start)
## an alternative would be to start with some finite value for yinit but
## then it will have to age before it is infectious

mout <- dede(y = yinit, 
		times = times, 
		func = seir, 
		parms = NULL)

# we have integrated the new infections and calculated the
# infectious (after LP) and removed (after LP+IP) with the
# lag function
# and the latent ones by subtraction

# output file for bit-mapped graphics
#jpeg(filename="seir%02d.jpeg", width=1000, height=720, units="px")
# output file for vector graphics
pdf("seir_desolve.pdf")

##-----------------------------
## display, plot results
##-----------------------------
matplot(mout[,1], mout[,c(2,4,5,6)], type = "l", lwd = 2, lty = 1,
  xlab="time", ylab="number",main = "SEIR model",col=c(1,2,3,4))
matplot(mout[,1], mout[,2], type = "l", lwd = 2, lty = 1,
  xlab="time", ylab="total",main = "SEIR model total infected",col=1)
matplot(mout[,1], mout[,4], type = "l", lwd = 2, lty = 1,
  xlab="time", ylab="Latent", main = "SEIR model latent",col=2)
matplot(mout[,1], mout[,5], type = "l", lwd = 2, lty = 1,
  xlab="time", ylab="Infectious", main = "SEIR model infectious",col=3)
matplot(mout[,1], mout[,6], type = "l", lwd = 2, lty = 1,
  xlab="time", ylab="Removed",  main = "SEIR model removed",col=4)
